*====================================================================
* This component manages an index on Outlook items. Additionally,
* contacts are managed. A list of items linked to a contact can be
* requested. Index data is stored in a database.
*====================================================================
Define Class OutlookIndex as Session

	Datasession = 2

	oNameSpace = NULL
	

*====================================================================
* Automation is used to access Outlook. We keep a reference to 
* Outlook all the time
*====================================================================
Procedure Init

	Local loOutlook as Outlook.Application
	loOutlook = CreateObject("Outlook.Application")
	This.oNameSpace = m.loOutlook.GetNamespace("MAPI")

Endproc 


*====================================================================
* When releasing the object, we free our references
*====================================================================
Procedure Destroy
	
	This.oNameSpace = NULL	

Endproc


*====================================================================
* Closes the current index
*====================================================================
Procedure CloseIndex
	
	If DBused("OLIndex")
		Set Database To OLIndex
		Close Tables
		Close Databases
	Endif 

EndProc


*====================================================================
* Opens an index database
*====================================================================
Procedure OpenIndex
LParameter tcDirectory

	This.CloseIndex()	
	Open Database (Addbs(m.tcDirectory)+"OLIndex.DBC")
	USE Folder In 0 Shared
	USE FolderToIndex In 0 Shared
	USE Item In 0 Shared
	USE IndexList in 0 Shared
	USE Word In 0 Shared
	
EndProc


*====================================================================
* Creates an empty database
*====================================================================
Procedure CreateIndex
LParameter tcDirectory

	*------------------------------------------------------------------
	* Create the Database
	*------------------------------------------------------------------
	This.CloseIndex()
	Create Database (Addbs(m.tcDirectory)+"OLIndex.DBC")
	
	*------------------------------------------------------------------
	* FLD.DBF: Contains a list of all folders and their relationship.
	*------------------------------------------------------------------
	Create Table (Addbs(m.tcDirectory)+"FLD") Name Folder ( ;
		FLD_ID I AutoInc Primary Key, ;
		FLD_StoreID C(166), ;
		FLD_EntryID C(48), ;
		FLD_ID1 I, ;
		FLD_Name C(60) ;
	)
	Index On FLD_StoreID+FLD_EntryID Tag FolderID
	
	*------------------------------------------------------------------
	* FTI.DBF: Contains the folders that should be indexed. The current
	* version doesn't allow to automatically add all subfolders.
	*------------------------------------------------------------------
	Create Table (Addbs(m.tcDirectory)+"FTI") Name FolderToIndex ( ;
		FTI_ID I AutoInc Primary Key, ;
		FLD_ID I ;
	)
	
	*------------------------------------------------------------------
	* ITM.DBF: Contains details about all items. Most fields duplicate
	* data in Outlook. The purpose is to be able to display a result
	* set fairly quickly without Automation to Outlook
	*------------------------------------------------------------------
	Create Table (Addbs(m.tcDirectory)+"ITM") Name Item ( ;
		ITM_ID I AutoInc Primary Key, ;
		ITM_EntryID C(48), ;
		FLD_ID I, ;
		ITM_Subject C(100), ;
		ITM_CreationTime T, ;
		ITM_Class N(2), ;
		ITM_LastModificationTime T ;
	)
	Index On ITM_EntryID+Str(FLD_ID) Tag ItemID
	
	*------------------------------------------------------------------
	* IDX.DBF: Many-to-many list that links Items and words together.
	*------------------------------------------------------------------
	Create Table (Addbs(m.tcDirectory)+"IDX") Name IndexList ( ;
		IDX_ID I AutoInc Primary Key, ;
		ITM_ID I, ;
		WRD_ID I ;
	)
	Index On ITM_ID Tag ITM_ID
	Index On WRD_ID Tag WRD_ID
	
	*------------------------------------------------------------------
	* WRD.DBF: All words in a normalized version. This table also con-
	* tains the ContactItem EntryId's when an object has been linked
	* to another. By putting them into a single table, we can more 
	* easily search for everything that deals with a contact.
	*------------------------------------------------------------------
	Create Table (Addbs(m.tcDirectory)+"WRD") Name Word ( ;
		WRD_ID I AutoInc Primary Key, ;
		WRD_Word C(48) ;
	)
	Index On WRD_Word Tag Word

	*------------------------------------------------------------------
	* Reopen all tables in shared mode
	*------------------------------------------------------------------
	This.CloseIndex()
	This.OpenIndex( m.tcDirectory )
	
Endproc


*====================================================================
* Updates the list of folders in the NAMESPACE. Folders that do not
* exist remain in the list, in case there was a problem with the 
* folder when it was opened.
*====================================================================
Procedure UpdateFolder
	
	This.UpdateFolderCollection( This.oNameSpace.Folders, 0 )

Endproc


*====================================================================
* This method is internally called by UpdateFolder and calls itself
* for each contained folder. A folder is identified by its StoreID.
*====================================================================
Protected Procedure UpdateFolderCollection
LParameter toFolders, tnParentID

	Local loFolder as Outlook.MAPIFolder, llExist
	For each loFolder in toFolders
		llExist = Seek( ;
			m.loFolder.StoreID+m.loFolder.EntryID, ;
			"Folder","FolderID" ;
		)
		If not m.llExist
			Append Blank in Folder
		Endif 
		Replace in Folder ;
			FLD_StoreID with m.loFolder.StoreID, ;
			FLD_EntryID with m.loFolder.EntryID, ;
			FLD_ID1 with m.tnParentID, ;
			FLD_Name with m.loFolder.Name
		If m.loFolder.Folders.Count > 0
			This.UpdateFolderCollection(m.loFolder.Folders,Folder.FLD_ID)
		Endif 
	Endfor 

EndProc


*====================================================================
* Each item is converted into a single string. Only this string is
* then indexed.
*====================================================================
Protected Procedure ItemToText
LParameter toItem

	Local lcText, loLink, loItem
	lcText = m.toItem.Subject + " " + m.toItem.Body + " " + ;
		m.toItem.Categories
	For each loLink in toItem.Links
		If Type("loLink.Item") == "O"
			loItem = loLink.Item
			lcText = m.lcText + " " + ;
				loItem.CompanyName + " " + ;
				loItem.FullName + " " + ;
				loItem.FileAs + " " + ;
				loItem.EntryID
		Else
			lcText = m.lcText + " " + m.loLink.Name
		Endif 
	endfor

Return m.lcText


*====================================================================
* Adds the text of a specified item to the index table. 
*====================================================================
Protected Procedure AddTextToIndex
LParameter tcText, tnITM_ID

	*------------------------------------------------------------------
	* WE have a local cursor here to avoid duplicate entries. It is 
	* faster to search this local cursor than scan the IDX table.
	*------------------------------------------------------------------
	Create Cursor curIDX ( WRD_ID I )
	Index on WRD_ID Tag WRD_ID
	
	*------------------------------------------------------------------
	* For each word, we determine its WRD_ID or add it, if its not in
	* the global word table. After that we link it to the item. 
	* GetWordNum() returns a couple of characters as words that we
	* don't need.
	*------------------------------------------------------------------
	Local lnWord, lcWord
	For lnWord = 1 to GetWordCount(m.tcText)
		lcWord = Chrtran( ;
			GetWordNum(m.tcText,m.lnWord), ;
			Chr(9)+Chr(13)+Chr(10)+Chr(34)+" !#$%&'()*+,-./:;<=>?[\]"+;
			"^`{|}~"+;
			"", ;
			"" ;
		)
		If Empty(m.lcWord)
			Loop
		Endif 
		lcWord = PadR(Lower(Alltrim(m.lcWord)),48)
		If not Seek(m.lcWord,"Word","Word")
			Insert into Word (WRD_Word) Values (m.lcWord)
		Endif
		If not Seek(Word.WRD_ID,"curIDX")
			Insert into IndexList (ITM_ID,WRD_ID) Values ( ;
				m.tnITM_ID, ;
				Word.WRD_ID ;
			)
			Insert into curIDX Values (Word.WRD_ID)
		Endif 
	Endfor

EndProc


*====================================================================
* Creates an entry for an item.
*====================================================================
Procedure IndexItem
LParameter toItem, tnFLD_ID

	*------------------------------------------------------------------
	* Determine if the item has changed or needs to be created
	*------------------------------------------------------------------
	Local llChanged, llNew
	If Seek(toItem.EntryID+Str(m.tnFLD_ID),"Item","ItemID")
		If Item.ITM_LastModificationTime == m.toItem.LastModificationTime
			llChanged = .F.
		Else
			llChanged = .T.
		Endif 
		llNew = .F.
	Else 
		Append blank in Item
		llChanged = .T.
		llNew = .T.
	Endif
	
	*------------------------------------------------------------------
	* Update the item entry
	*------------------------------------------------------------------
	If m.llChanged
		Replace in Item ;
			ITM_EntryID with m.toItem.EntryID, ;
			FLD_ID with m.tnFLD_ID, ;
			ITM_Subject with m.toItem.Subject, ;
			ITM_CreationTime with m.toItem.CreationTime, ;
			ITM_Class with m.toItem.Class, ;
			ITM_LastModificationTime with m.toItem.LastModificationTime
	Endif 
	
	*------------------------------------------------------------------
	* If we update the index, remove old records. Currently we are
	* only deleting them. In the future we might recycle deleted 
	* records.
	*------------------------------------------------------------------
	If not m.llNew
		Delete in IndexList For ITM_ID == Item.ITM_ID
	Endif 

	*------------------------------------------------------------------
	* Add the item to the index
	*------------------------------------------------------------------
	Local lcText
	If m.llChanged
		lcText = This.ItemToText( m.toItem )
		This.AddTextToIndex( m.lcText, Item.ITM_ID )
	Endif 

Endproc 	


*====================================================================
* Updates the index for all items in a folder
*====================================================================
Procedure IndexFolder
LParameter tnFLD_ID

	Local loItem, loFolder
	loFolder = This.GetFolderReference( m.tnFLD_ID )
	If Type("m.loFolder") == "O"
		For each loItem in loFolder.Items
			Wait Window NOWAIT loItem.Subject
			This.IndexItem( m.loItem, m.tnFLD_ID)
		Endfor
	Endif 

Endproc


*====================================================================
* Returns a reference to the folder
*====================================================================
Procedure GetFolderReference
LParameter tnFLD_ID

	Local loFolder
	If Seek(m.tnFLD_ID,"Folder","FLD_ID")
		loFolder = This.oNameSpace.GetFolderFromID( ;
			Folder.FLD_EntryID, ;
			Folder.FLD_StoreID ;
		)
	Else
		loFolder = NULL
	Endif 

Return m.loFolder


*====================================================================
* Returns a reference to an item
*====================================================================
Procedure GetItemReference
LParameter tnITM_ID

	Local loItem, loFolder
	If     Seek(m.tnITM_ID,"Item","ITM_ID") ;
	   and Seek(Item.FLD_ID,"Folder","FLD_ID")
		loItem = This.oNameSpace.GetItemFromID( ;
			Item.ITM_EntryID, ;
			Folder.FLD_StoreID ;
		)
	Else
		loItem = NULL
	Endif 
	
Return m.loItem	


*====================================================================
* Updates all index in the selected folders
*====================================================================
Procedure ReIndex

	Select FolderToIndex
	If Reccount() == 0
		MessageBox("Please add entries to the FolderToIndex tables.")
	EndIf
	Scan 
		This.IndexFolder( FolderToIndex.FLD_ID )
	Endscan 	

Endproc


*====================================================================
* Displays the specified item
*====================================================================
Procedure ShowItem
LParameter tnITM_ID

	Local loItem
	If Seek(m.tnITM_ID,"Item","ITM_ID")
		loItem = This.GetItemReference( m.tnITM_ID )
	Endif
	loItem.GetInspector.Activate()

EndProc


Enddefine 

