*========================================================================================
* Encapsulates call to the Google Translate API v2
*========================================================================================
Define Class GoogleTranslate as Custom

	#include Acodey.h

	*--------------------------------------------------------------------------------------
	* Enter your private API key here
	*--------------------------------------------------------------------------------------
	cApiKey = ""
	
	*--------------------------------------------------------------------------------------
	* External dependencies
	*
	* These should be managed through a service broker concept, the EvalRef method or by 
	* using an injection technology.
	*--------------------------------------------------------------------------------------
	oHttp = NULL
	oJson = NULL
	oEncode = NULL

*========================================================================================
* Initialize GoogleTranslate
*
* Requirements: JSON.FLL must have been loaded prior to instantiating GoogleTranslate
*========================================================================================
Procedure Init

	*--------------------------------------------------------------------------------------
	* The class relies on the JSON.FLL. Because we can't embedd FLLs into projects and 
	* because we can't generically control the current directory and the location of 
	* helper FLLS, we insist that the library must be loaded prior.
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert "JSON.FLL" $ Upper(Set("Library"))
	#ENDIF

	*--------------------------------------------------------------------------------------
	* You can use a different Http client by assigning an object to oHttp in the Init 
	* event of a subclass before calling DODEFAULT()
	*--------------------------------------------------------------------------------------
	If IsNull( This.oHttp )
		This.oHttp = NewObject("cHttpClient", "cHttpClient.prg")
		This.oHttp.UseSsl()
	EndIf 
	If IsNull( This.oEncode )
		This.oEncode = NewObject("CHttpEncoding", "cHttpClient.prg")
	EndIf
	
	*--------------------------------------------------------------------------------------
	* You can use a different JSON parser by assigning an object before calling 
	* DODEFAULT().
	*--------------------------------------------------------------------------------------
	If IsNull( This.oJson )
		This.oJson = NewObject("Json", "Json/Json.prg")
	EndIf 
	
EndProc

*========================================================================================
* Translates a string
*========================================================================================
Procedure Translate (tcString, tcFrom, tcTo)

	*--------------------------------------------------------------------------------------
	* Assemble Google Translate query. Parameter string must be URL encoded
	*--------------------------------------------------------------------------------------
	Local lcData, lcString
	lcString = This.oEncode.UrlEncode (m.tcString)
	lcData = "" ;
		+"key="+This.cApiKey+"&" ;
		+"source="+m.tcFrom+"&" ;
		+"target="+m.tcTo+"&" ;
		+"q="+m.lcString
	
	*--------------------------------------------------------------------------------------
	* Server responds with an object
	*--------------------------------------------------------------------------------------
	Local loResult, lcTranslation
	loResult = This.SendRequest(m.lcData)
	Do case
	Case PemStatus(m.loResult, "data", 5)
		lcTranslation = m.loResult.data.Translations[1].translatedText
	Case PemStatus(m.loResult, "error", 5)
		lcTranslation = "error: "+m.loResult.Error.Message
	Otherwise
		lcTranslation = "error: "
	EndCase
	
	*--------------------------------------------------------------------------------------
	* Google uses UTF-8
	*--------------------------------------------------------------------------------------
	lcTranslation = Strconv(m.lcTranslation,11)

Return m.lcTranslation


*========================================================================================
* POSTs a request and returns the result object
*========================================================================================
Protected Procedure SendRequest (tcData)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcData) == T_CHARACTER
		Assert not Empty(m.tcData)
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Send request
	*--------------------------------------------------------------------------------------
	Local lcResponse
	If This.oHttp.Connect ("www.googleapis.com")
		This.oHttp.cHeaders = ;
			+"Content-Type: application/x-www-form-urlencoded"+Chr(13)+Chr(10);
			+"X-HTTP-Method-Override: GET"
		lcResponse = This.oHttp.Post( ;
			"https://www.googleapis.com/language/translate/v2", ;
			m.tcData ;
		)
		This.oHttp.Disconnect ()
	Else
		lcResponse = NULL
	EndIf

	*--------------------------------------------------------------------------------------
	* Handle response
	*--------------------------------------------------------------------------------------
	Local loResult
	If IsNull(m.lcResponse) 
		loResult = CreateObject("Empty")
	else
		loResult = This.ParseResponse (m.lcResponse, This.oHttp.cStatusCode)
	EndIf
	
Return m.loResult

*========================================================================================
* Server responds with either an HTTP error message or an error object in JSON 
* notation.
*========================================================================================
Protected FUNCTION ParseResponse (tcData, tcStatus)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcData) == T_CHARACTER
		Assert Vartype(m.tcStatus) == T_CHARACTER
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Responses with a status code of 200 must be JSON, all other can be HTML. oAuth 2.0 
	* defines a number of error messages that providers must or should return in case of 
	* an error. Those are always JSON objects. Error conditions not covered by these 
	* specific situations, such as missing parameters, invalid form data, missing content
	* types, and the like, seem to trigger HTML responses at least for Google.
	*--------------------------------------------------------------------------------------
	Local loResponse
	If m.tcStatus == "200"
		loResponse = This.ParseJsonResponse (m.tcData)
	Else
		If Left( Alltrim(m.tcData), 1) == "{"
			loResponse = This.ParseJsonResponse (m.tcData)
		Else
			loResponse = This.TurnHtmlIntoErrorObject (m.tcData)
		EndIf
	EndIf
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loResponse) == T_OBJECT
	#ENDIF

Return m.loResponse

*========================================================================================
* Turns a JSON string into an object. This code uses the JSON parser from Craig Boyd.
* The JSON parser has been modified. Modifications are tagged with "* CW:".
*
* http://www.sweetpotatosoftware.com/spsblog/2008/12/19/VisualFoxproJSONClassUpdate.aspx
*========================================================================================
Protected FUNCTION ParseJsonResponse (tcJson)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcJson) == T_CHARACTER
		Assert not Empty(m.tcJson)
		Assert Left(Alltrim(m.tcJson),1) == "{"
		Assert Right(Alltrim(Chrtran(m.tcJson,Chr(13)+Chr(10),"")),1) == "}"
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Parse the object
	*--------------------------------------------------------------------------------------
	Local loObj
	loObj = This.oJson.Parse (m.tcJson)
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loObj) == T_OBJECT
	#ENDIF
	
Return m.loObj

*========================================================================================
* Some errors return HTML messages instead of JSON objects
*========================================================================================
Protected FUNCTION TurnHtmlIntoErrorObject (tcHtml)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcHtml) == T_CHARACTER
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Extract a meaningful error message from the HTML text.
	*--------------------------------------------------------------------------------------
	Local lcMsg, loObj
	lcMsg = StrExtract(m.tcHtml, "<title>", "</title>", 1, 1)
	loObj = CreateObject("Empty")
	AddProperty(m.loObj, "error", m.lcMsg)
	AddProperty(m.loObj, "html", m.tcHtml)
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loObj) == T_OBJECT
		Assert PemStatus(m.loObj, "error", 5)
	#ENDIF

Return m.loObj

EndDefine 


