*========================================================================================
* Generic oAuth2 Client
*
* This class is meant to be subclassed for every provider. Please provide 
* configuration values and an implementation for DoGetRequestToken.
*========================================================================================
Define Class cOAuth2 as Custom

	#include Acodey.h

	*--------------------------------------------------------------------------------------
	* oAuth2 client configuration section. These values need to be changed based on 
	* the developer's registration with the resource provider.
	*--------------------------------------------------------------------------------------
	cClientId = ""
	cClientSecret = ""
	cScope = ""
	
	*--------------------------------------------------------------------------------------
	* oAuth2 provider configuration section. For each resource provider such as 
	* Google, Facebook, Twitter, you need a subclass that implements the specifcs 
	* of each provider. Change these properties in the subclass.
	*--------------------------------------------------------------------------------------
	cRedirectUri = ""
	cServer = ""
	cUrlToken = ""
	cUrlAuthorize = ""
	
	*--------------------------------------------------------------------------------------
	* Token values. Persist these values at runtime. These values are provider and 
	* client specific. 
	*--------------------------------------------------------------------------------------
	cAccessToken = ""
	cRefreshToken = ""
	
	*--------------------------------------------------------------------------------------
	* External dependencies
	*
	* These should be managed through a service broker concept, the EvalRef method 
	* or by using an injection technology.
	*--------------------------------------------------------------------------------------
	oHttp = NULL
	oJson = NULL
	
*========================================================================================
* Initialize oAuth2 client
*
* Requirements: JSON.FLL must have been loaded prior to instantiating the oAuth2 
*               client.
*========================================================================================
Procedure Init

	*--------------------------------------------------------------------------------------
	* The client relies on the JSON.FLL. Because we can't embedd FLLs into projects 
	* and because we can't generically control the current directory and the location
	* of helper FLLS, we insist that the library must be loaded prior.
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert "JSON.FLL" $ Upper(Set("Library"))
	#ENDIF

	*--------------------------------------------------------------------------------------
	* You can use a different Http client by assigning an object to oHttp in the Init 
	* event of a subclass before calling DODEFAULT()
	*--------------------------------------------------------------------------------------
	If IsNull( This.oHttp )
		This.oHttp = NewObject("cHttpClient", "cHttpClient.prg")
		This.oHttp.UseSsl()
	EndIf 
	
	*--------------------------------------------------------------------------------------
	* You can use a different JSON Parser by assigning an object before calling 
	* DODEFAULT().
	*--------------------------------------------------------------------------------------
	If IsNull( This.oJson )
		This.oJson = NewObject("Json", "Json/Json.prg")
	EndIf 
	
EndProc

*========================================================================================
* Returns the access token for use in an HTTP request.
*========================================================================================
Function GetAccessToken ()

	*--------------------------------------------------------------------------------------
	* When an access token becomes invalid you must call the InvalidateAccessToken 
	* method. Otherwise this method keeps returning the previous access token.
	*--------------------------------------------------------------------------------------
	If not Empty(This.cAccessToken)
		Return This.cAccessToken
	EndIf 
	
	*--------------------------------------------------------------------------------------
	* If we have a refresh token request a new access token.
	*--------------------------------------------------------------------------------------
	If not Empty(This.cRefreshToken)
		This.RefreshAccessToken ()
	EndIf 
	If not Empty(This.cAccessToken)
		Return This.cAccessToken
	EndIf 
	
	*--------------------------------------------------------------------------------------
	* If neither worked we ask the user again for permission
	*--------------------------------------------------------------------------------------
	This.RequestAccessToken ()
	If not Empty(This.cAccessToken)
		Return This.cAccessToken
	EndIf 
		
Return ""

*========================================================================================
* Uses the refresh token to request a new access token
*========================================================================================
Protected Procedure RefreshAccessToken 

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert not Empty(This.cRefreshToken)
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Send a request using the refresh token
	*--------------------------------------------------------------------------------------
	This.SendRequest ("" ;
		+"client_id="+This.cClientId+"&" ;
		+"client_secret="+This.cClientSecret+"&" ;
		+"refresh_token="+This.cRefreshToken+"&" ;
		+"grant_type=refresh_token" ;
	)

EndProc

*========================================================================================
* POTSs an authorization request and returns the response obejct
*========================================================================================
Protected Procedure SendRequest (tcData)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcData) == T_CHARACTER
		Assert not Empty(m.tcData)
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Send request
	*--------------------------------------------------------------------------------------
	Local lcResponse
	If This.oHttp.Connect (This.cServer)
		This.oHttp.cHeaders = "Content-Type: application/x-www-form-urlencoded"
		lcResponse = This.oHttp.Post( ;
			"https://"+This.cServer+"/"+This.cUrlToken, m.tcData)
		This.oHttp.Disconnect ()
	Else
		lcResponse = NULL
	EndIf
	
	*--------------------------------------------------------------------------------------
	* Handle response
	*--------------------------------------------------------------------------------------
	Local loResult
	If not IsNull(m.lcResponse) 
		loResult = This.ParseResponse (m.lcResponse, This.oHttp.cStatusCode)
		If PemStatus(m.loResult, "access_token", 5)
			This.cAccessToken = m.loResult.access_token
		EndIf
		If PemStatus(m.loResult, "refresh_token", 5)
			This.cRefreshToken = m.loResult.refresh_token
		EndIf
	EndIf
	
EndProc

*========================================================================================
* Server responds with either an HTTP error message or an error object in JSON 
* notation.
*========================================================================================
Protected FUNCTION ParseResponse (tcData, tcStatus)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcData) == T_CHARACTER
		Assert Vartype(m.tcStatus) == T_CHARACTER
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Responses with a status code of 200 must be JSON, all other can be HTML. oAuth 
	* 2.0 defines a number of error messages that providers must or should return in 
	* case of an error. Those are always JSON objects. Error conditions not covered
	* by these specific situations, such as missing parameters, invalid form data, 
	* missing content types, and the like, seem to trigger HTML responses at least 
	* for Google.
	*--------------------------------------------------------------------------------------
	Local loResponse
	If m.tcStatus == "200"
		loResponse = This.ParseJsonResponse (m.tcData)
	Else
		If Left( Alltrim(m.tcData), 1) == "{"
			loResponse = This.ParseJsonResponse (m.tcData)
		Else
			loResponse = This.TurnHtmlIntoErrorObject (m.tcData)
		EndIf
	EndIf
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loResponse) == T_OBJECT
	#ENDIF

Return m.loResponse

*========================================================================================
* Turns a JSON string into an object. This code uses the JSON parser from Craig Boyd.
* The JSON parser has been modified. Modifications are tagged with "* CW:".
*
* http://www.sweetpotatosoftware.com/spsblog/2008/12/19/VisualFoxproJSONClassUpdate.aspx
*========================================================================================
Protected FUNCTION ParseJsonResponse (tcJson)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcJson) == T_CHARACTER
		Assert not Empty(m.tcJson)
		Assert Left(Alltrim(m.tcJson),1) == "{"
		Assert Right(Alltrim(m.tcJson),1) == "}"
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Parse the object
	*--------------------------------------------------------------------------------------
	Local loObj
	loObj = This.oJson.Parse (m.tcJson)
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loObj) == T_OBJECT
	#ENDIF
	
Return m.loObj

*========================================================================================
* Some errors return HTML messages instead of JSON objects
*========================================================================================
Protected FUNCTION TurnHtmlIntoErrorObject (tcHtml)

	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcHtml) == T_CHARACTER
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* Extract a meaningful error message from the HTML text.
	*--------------------------------------------------------------------------------------
	Local lcMsg, loObj
	lcMsg = StrExtract(m.tcHtml, "<title>", "</title>", 1, 1)
	loObj = CreateObject("Empty")
	AddProperty(m.loObj, "error", m.lcMsg)
	AddProperty(m.loObj, "html", m.tcHtml)
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.loObj) == T_OBJECT
		Assert PemStatus(m.loObj, "error", 5)
	#ENDIF

Return m.loObj

*========================================================================================
* Prompt the user for permission and obtains an access token
*========================================================================================
Protected Procedure RequestAccessToken 
	
	Local lcRequestToken
	m.lcRequestToken = This.GetRequestToken (This.GetAuthorizationUrl ())
	If not Empty(m.lcRequestToken)
		This.SendRequest ("" ;
			+"client_id="+This.cClientId+"&" ;
			+"client_secret="+This.cClientSecret+"&" ;
			+"code="+m.lcRequestToken+"&" ;
			+"redirect_uri="+This.cRedirectUri+"&" ;
			+"grant_type=authorization_code" ;
		)
			EndIf

EndProc

*========================================================================================
* Wraps DoGetRequestToken
*========================================================================================
Protected Function GetRequestToken (tcUrl)
	
	*--------------------------------------------------------------------------------------
	* Assertions
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.tcUrl) == T_CHARACTER
	#ENDIF
	
	*--------------------------------------------------------------------------------------
	* You must override DoGetRequestToken in a subclass.
	*--------------------------------------------------------------------------------------
	Local lcRequestToken
	lcRequestToken = This.DoGetRequestToken (m.tcUrl)
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.lcRequestToken) == T_CHARACTER
	#ENDIF

Return m.lcRequestToken

*========================================================================================
* Insert the code to obtain the request token from the user. Return an empty string
* if permissions are denied.
*========================================================================================
Function DoGetRequestToken (tcUrl)
Return ""	

*========================================================================================
* Returns the authorization request Url.
*========================================================================================
Protected Function GetAuthorizationUrl

	*--------------------------------------------------------------------------------------
	* Request authorization
	*--------------------------------------------------------------------------------------
	Local lcUrl
	lcUrl = "https://"+This.cServer+"/"+This.cUrlAuthorize+"?" ;
		+"client_id="+This.cClientId+"&" ;
		+"redirect_uri="+This.cRedirectUri+"&" ;
		+"scope="+This.cScope+"&" ;
		+"response_type=code"
	
	*--------------------------------------------------------------------------------------
	* Post-Condition
	*--------------------------------------------------------------------------------------
	#IF __DEBUGLEVEL >= __DEBUG_REGULAR
		Assert Vartype(m.lcUrl) == T_CHARACTER
	#ENDIF
	
Return m.lcUrl

*========================================================================================
* Call this method when the access token has become invalid and webAPI calls return 
* with access denied errors.
*========================================================================================
Procedure InvalidateAccessToken
	This.cAccessToken = ""
EndProc

EndDefine 


