*====================================================================
* This procedure library contains functions to convert VFP data types
* to strings that can be used to assemble structures and whenever you
* have to convert between different data types.
*
* Most of these function use the RtlMoveMemory() function to copy the
* binary representation into a string and back.
*
* In VFP 6 Service Pack 3 is required!
*====================================================================


*====================================================================
* Converts a floating point variable (DOUBLE, REAL) into a binary 
* string.
*====================================================================
Procedure APIDoubleToChar
LParameter tnDouble

	Local lcString
	lcString = Space(8)
	Declare RtlMoveMemory in Win32API as __DoubleToChar ;
		String@, Double@, Integer
	__DoubleToChar( @lcString, @tnDouble, 8 )
	
Return m.lcString


*====================================================================
* Converts a floating point variable (SINGLE) into a binary string.
*====================================================================
Procedure ApiSingleToChar
LParameter tnSingle

	Local lcString
	lcString = Space(4)
	Declare RtlMoveMemory in Win32API as __SingleToChar ;
		String@, Single@, Integer
	__SingleToChar( @lcString, @tnSingle, 4 )
	
Return m.lcString


*====================================================================
* Converts a 32-bit integer variable (LONG) into a binary string.
*====================================================================
Procedure APILongToChar
LParameter tnLong

	Local lcString
	lcString = Space(4)
	Declare RtlMoveMemory in Win32API as __LongToChar ;
		String@, Long@, Integer
	__LongToChar( @lcString, @tnLong, 4 )
	
Return m.lcString


*====================================================================
* Converts a 16-bit integer variable (WORD, SHORT) into a binary 
* string.
*====================================================================
Procedure APIShortToChar
LParameter tnShort

	Local lcString
	lcString = Space(2)
	Declare RtlMoveMemory in Win32API as __ShortToChar ;
		String@, Short@, Integer
	__ShortToChar( @lcString, @tnShort, 2 )
	
Return m.lcString


*====================================================================
* Converts a 8-bit integer variable (BYTE) into a binary string.
*
* First of all we convert negative value to positive values and
* reduce the range of the byte to 0..255. The MOD() function
* does exactly this. It might seem to behave strangely, because
* MOD(-1,256) is 255, but that's exactly how negative values are
* treated in binary representation. After that we can use CHR() to
* convert the byte into a string.
*====================================================================
Procedure APIByteToChar
LParameter tnByte

	Local lcString
	lcString = Chr( Mod(m.tnByte,256) )
		
Return m.lcString


*====================================================================
* Converts a binary string into a floating point variable (DOUBLE, 
* REAL)
* BUG: All versions of VFP don't set the number of digits following
*      the decimal point correctly. If you convert 123.456 to a
*      string and back and print the result, you get 123. Internally,
*      however, the correct value is stored. 
*====================================================================
Procedure APICharToDouble
LParameter tcString

	Local lnDouble
	lnDouble = 0
	Declare RtlMoveMemory in Win32API as __CharToDouble ;
		Double@, String@, Integer
	__CharToDouble( @lnDouble, @tcString, 8 )
	lnDouble = m.lnDouble - 0.0000000000000000000
	
Return m.lnDouble


*====================================================================
* Converts a binary string into a floating point variable (SINGLE)
* BUG: All versions of VFP don't set the number of digits following
*      the decimal point correctly. If you convert 123.456 to a
*      string and back and print the result, you get 123. Internally,
*      however, the correct value is stored. 
*====================================================================
Procedure APICharToSingle
LParameter tcString

	Local lnSingle
	lnSingle = 0
	Declare RtlMoveMemory in Win32API as __CharToSingle ;
		Single@, String@, Integer
	__CharToSingle( @lnSingle, @tcString, 4 )
	lnSingle = m.lnSingle - 0.0000
	
Return m.lnSingle


*====================================================================
* Converts a binary string into a 32-bit unsigned integer.
*====================================================================
Procedure APICharToLong
LParameter tcString

	Local lnLong
	lnLong = 0
	Declare RtlMoveMemory in Win32API as __CharToLong ;
		Long@, String@, Integer
	__CharToLong( @lnLong, @tcString, 4 )
	If m.lnLong < 0
		lnLong = m.lnLong + 2^32
	Endif
	
Return m.lnLong


*====================================================================
* Converts a binary string into a 32-bit signed integer.
*====================================================================
Procedure APICharToSLong
LParameter tcString

	Local lnLong
	lnLong = 0
	Declare RtlMoveMemory in Win32API as __CharToSLong ;
		Long@, String@, Integer
	__CharToSLong( @lnLong, @tcString, 4 )
	
Return m.lnLong


*====================================================================
* Converts a binary string into a 16-bit unsigned integer.
* BUG: VFP 5 can't handle SHORT@ properly. 
*====================================================================
Procedure APICharToShort
LParameter tcString

	Local lnShort
	lnShort = 0
	If Type("Version(4)") == "C"
		Declare RtlMoveMemory in Win32API as __CharToShort ;
			Short@, String@, Integer
	Else
		Declare RtlMoveMemory in Win32API as __CharToShort ;
			Integer@, String@, Integer
	Endif
	__CharToShort( @lnShort, @tcString, 2 )
	If Type("Version(4)") == "C"
		If m.lnShort < 0
			lnShort = m.lnShort + 2^16
		Endif
	Endif
	
Return m.lnShort


*====================================================================
* Converts a binary string into a 16-bit signed integer.
* BUG: VFP 5 can't handle SHORT@ properly. 
*====================================================================
Procedure APICharToSShort
LParameter tcString

	Local lnShort
	lnShort = 0
	If Type("Version(4)") == "C"
		Declare RtlMoveMemory in Win32API as __CharToSShort ;
			Short@, String@, Integer
	Else
		Declare RtlMoveMemory in Win32API as __CharToSShort ;
			Integer@, String@, Integer
	Endif
	__CharToSShort( @lnShort, @tcString, 2 )
	If not Type("Version(4)") == "C"
		If m.lnShort >= 2^15
			lnShort = m.lnShort - 2^16
		Endif
	Endif
	
Return m.lnShort


*====================================================================
* Convert a binary string into a 8-bit unsigned integer
*====================================================================
Procedure APICharToByte
LParameter tcString 

	Local lnByte
	lnByte = Asc( m.tcString )

Return m.lnByte


*====================================================================
* Convert a binary string into a 8-bit signed integer
*====================================================================
Procedure APICharToSByte
LParameter tcString 

	Local lnByte
	lnByte = Asc( m.tcString )
	If m.lnByte > 127
		lnByte = m.lnByte - 256
	Endif

Return m.lnByte
