// Hilfsbibliothek zum Laden von MTDLL COM Servern in eigene Threads

#include "stdafx.h"


//===================================================================
// Parameterstruktur fr CreateObjectWithString
//===================================================================
struct CREATETHREADWITHSTRING {
	CString strClass;
	CString strMethod;
	CString strParameter;
} ;

DWORD WINAPI CreateObjectWithString( LPVOID pVoid )
{

	//-----------------------------------------------------------------
	// Vorbereitungen treffen
	//-----------------------------------------------------------------
	CREATETHREADWITHSTRING *pParam = (CREATETHREADWITHSTRING *) pVoid;
	BOOL bOK = TRUE;

	//-----------------------------------------------------------------
	// Ein neues Apartment anlegen
	//-----------------------------------------------------------------
	if( bOK ) {
		HRESULT hr = CoInitializeEx( NULL, COINIT_APARTMENTTHREADED );
		if( !(hr==S_OK || hr==S_FALSE) )
		{
			return FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Als Parameter erhalten wir nur die ProgID. Daher mssen wir nun
	// deren CLSID ermitteln
	//-----------------------------------------------------------------
	CLSID clsidMTDLL;
	if( bOK ) {
		BSTR bstrClass = pParam->strClass.AllocSysString();
		HRESULT hr = CLSIDFromString( bstrClass, &clsidMTDLL );
		SysFreeString( bstrClass );
		if( FAILED(hr) ) {
			bOK = FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Mit der CLSID knnen wir den Server jetzt laden
	//-----------------------------------------------------------------
	IDispatch *pdisp = NULL;
	if( bOK ) {
		HRESULT hr = CoCreateInstance( 
			clsidMTDLL, NULL, CLSCTX_ALL, IID_IDispatch, (LPVOID*) &pdisp
		);
		if( FAILED(hr) ) {
			pdisp = NULL;
			bOK = FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Jede Methode hat eine eindeutige ID. Diese ID ermitteln wir nun
	// fr den bergebenen Methodennamen.
	//-----------------------------------------------------------------
	DISPID dispid;
	if( bOK ) {
		BSTR bstrMethod = pParam->strMethod.AllocSysString();
		HRESULT hr = pdisp->GetIDsOfNames( 
			IID_NULL, &bstrMethod, 1, LOCALE_USER_DEFAULT, &dispid
		);
		SysFreeString( bstrMethod );
		if( FAILED(hr) ) {
			bOK = FALSE;
		};
	}

	//-----------------------------------------------------------------
	// Parameter an COM Server mssen extra aufbereitet werden. 
	// Achtung: Mehrere Parameter werden rckwrts angegeben. 0 ist der
	//          letzte Parameter.
	//-----------------------------------------------------------------
	VARIANTARG varg[1];
	BSTR bstrParameter = NULL;
	if( bOK ) {
		bstrParameter = pParam->strParameter.AllocSysString();
		varg[0].vt = VT_BSTR ;
		varg[0].bstrVal = bstrParameter;
	}

	//-----------------------------------------------------------------
	// Mit der DISPID und dem Parameter nun die Methode ausfhren.
	//-----------------------------------------------------------------
	if( bOK ) {
		DISPPARAMS dispparms = {varg, NULL, 1, 0};
		HRESULT hr = pdisp->Invoke( 
			dispid, 
			IID_NULL,
			LOCALE_USER_DEFAULT, 
			DISPATCH_METHOD, 
			&dispparms,
			NULL,
			NULL,
			NULL 
		);
		if( FAILED(hr) ) {
			bOK = FALSE;
		}
	}
	
	//-----------------------------------------------------------------
	// Wenn bstrParameter belegt wurde, diesen freigeben
	//-----------------------------------------------------------------
	if( bstrParameter != NULL ) {
		SysFreeString(bstrParameter);
	}

	//-----------------------------------------------------------------
	// Das VFP Objekt freigeben, sofern es fehlerfrei erzeugt werden
	// konnte.
	//-----------------------------------------------------------------
	if( pdisp != NULL ) {
		pdisp->Release();
	}

	//-----------------------------------------------------------------
	// Speicher fr den Parameter freigeben
	//-----------------------------------------------------------------
	delete pParam;

	//-----------------------------------------------------------------
	// COM Umgebung aufrumen und Thread beenden
	//-----------------------------------------------------------------
	CoUninitialize();

	return bOK;
}


//===================================================================
// Diese Funktion wird aus VFP Anwendungen aufgerufen, um einen neuen
// Thread zu erzeugen. An den Thread wird eine Zeichenkette berge-
// ben. Zurckgegeben wird die Thread ID und der Handle auf den
// Thread.
//
// Alle Zeichenketten sind Unicode. Sie mssen vor dem Aufruf mit
// STRCONV() konvertiert werden.
//===================================================================
__declspec(dllexport) HANDLE CreateThreadWithString( 
		LPWSTR lpszClass,
		LPWSTR lpszMethod,
		LPWSTR lpszParameter, 
		LPDWORD lpdwThreadId 
	)
{
	//-----------------------------------------------------------------
	// An einen Thread knnen wir nur einen Parameter bergeben. Daher
	// speichern wir alle Werte in einer Struktur. Diese Struktur 
	// mssen wir dynamisch allokieren, damit der Speicher noch verfg-
	// bar ist, wenn diese Funktion beendet wurde. Der Speicher wird
	// durch den Thread freigegeben.
	//-----------------------------------------------------------------
	CREATETHREADWITHSTRING *pParam;
	pParam = new CREATETHREADWITHSTRING;
	pParam->strClass = lpszClass;
	pParam->strMethod = lpszMethod;
	pParam->strParameter = lpszParameter;
	
	//-----------------------------------------------------------------
	// Thread erzeugen. Das Laden des COM Objektes passiert im neuen
	// Thread.
	//-----------------------------------------------------------------
	HANDLE hThread;
	DWORD dwThreadID;
	hThread = CreateThread( 
		NULL,	0,CreateObjectWithString, pParam,	0, &dwThreadID
	);
	*lpdwThreadId = dwThreadID;
	
	//-----------------------------------------------------------------
	// Falls der Thread nicht erzeugt werden konnte, mssen wir den
	// Parameter wieder freigeben.
	//-----------------------------------------------------------------
	if( hThread == NULL ) {
		delete pParam;
	}

	return hThread;
}


//===================================================================
// Parameterstruktur fr CreateThreadWithObject
//===================================================================
struct CREATETHREADWITHOBJECT {
	CString strClass;
	CString strMethod;
	LPSTREAM pStream;
} ;

DWORD WINAPI CreateThreadWithObject( LPVOID pVoid )
{

	//-----------------------------------------------------------------
	// Vorbereitungen treffen
	//-----------------------------------------------------------------
	CREATETHREADWITHOBJECT *pParam = (CREATETHREADWITHOBJECT *) pVoid;
	BOOL bOK = TRUE;
	
	//-----------------------------------------------------------------
	// Ein neues Apartment anlegen
	//-----------------------------------------------------------------
	if( bOK ) {
		HRESULT hr = CoInitializeEx( NULL, COINIT_APARTMENTTHREADED );
		if( !(hr==S_OK || hr==S_FALSE) )
		{
			return FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Der Stream muss in jedem Fall gelesen werden. Anschlieend haben
	// wir eine Objektreferenz.
	//-----------------------------------------------------------------
	IDispatch* pObject = NULL;
	if( bOK ) {
		HRESULT hr = CoGetInterfaceAndReleaseStream( 
			pParam->pStream,
			IID_IDispatch,
			(void **) &pObject
		);
		if( FAILED(hr) ) {
			bOK = FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Als Parameter erhalten wir nur die ProgID. Daher mssen wir nun
	// deren CLSID ermitteln
	//-----------------------------------------------------------------
	CLSID clsidMTDLL;
	if( bOK ) {
		BSTR bstrClass = pParam->strClass.AllocSysString();
		HRESULT hr = CLSIDFromString( bstrClass, &clsidMTDLL );
		SysFreeString( bstrClass );
		if( FAILED(hr) ) {
			bOK = FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Mit der CLSID knnen wir den Server jetzt laden
	//-----------------------------------------------------------------
	IDispatch *pdisp = NULL;
	if( bOK ) {
		HRESULT hr = CoCreateInstance( 
			clsidMTDLL, NULL, CLSCTX_ALL, IID_IDispatch, (LPVOID*) &pdisp
		);
		if( FAILED(hr) ) {
			pdisp = NULL;
			bOK = FALSE;
		}
	}

	//-----------------------------------------------------------------
	// Jede Methode hat eine eindeutige ID. Diese ID ermitteln wir nun
	// fr den bergebenen Methodennamen.
	//-----------------------------------------------------------------
	DISPID dispid;
	if( bOK ) {
		BSTR bstrMethod = pParam->strMethod.AllocSysString();
		HRESULT hr = pdisp->GetIDsOfNames( 
			IID_NULL, &bstrMethod, 1, LOCALE_USER_DEFAULT, &dispid
		);
		SysFreeString( bstrMethod );
		if( FAILED(hr) ) {
			bOK = FALSE;
		};
	}

	//-----------------------------------------------------------------
	// Parameter an COM Server mssen extra aufbereitet werden. 
	// Achtung: Mehrere Parameter werden rckwrts angegeben. 0 ist der
	//          letzte Parameter.
	//-----------------------------------------------------------------
	VARIANTARG varg[1];
	if( bOK ) {
		varg[0].vt = VT_DISPATCH ;
		varg[0].pdispVal = pObject;
	}

	//-----------------------------------------------------------------
	// Mit der DISPID und dem Parameter nun die Methode ausfhren.
	//-----------------------------------------------------------------
	if( bOK ) {
		DISPPARAMS dispparms = {varg, NULL, 1, 0};
		HRESULT hr = pdisp->Invoke( 
			dispid, 
			IID_NULL,
			LOCALE_USER_DEFAULT, 
			DISPATCH_METHOD, 
			&dispparms,
			NULL,
			NULL,
			NULL 
		);
		if( FAILED(hr) ) {
			bOK = FALSE;
		}
	}
	
	//-----------------------------------------------------------------
	// Das VFP Objekt freigeben, sofern es fehlerfrei erzeugt werden
	// konnte.
	//-----------------------------------------------------------------
	if( pdisp != NULL ) {
		pdisp->Release();
	}

	//-----------------------------------------------------------------
	// Beim bergeben des Objektes haben wir den Referenzzhler hht.
	// Dies machen wir nun rckgngig, damit das Objekt freigegeben
	// werden kann. Release() ist das Gegenteil von AddRef().
	//-----------------------------------------------------------------
	if( pObject != NULL ) {
		pObject->Release();
	}

	//-----------------------------------------------------------------
	// Speicher fr den Parameter freigeben
	//-----------------------------------------------------------------
	delete pParam;

	//-----------------------------------------------------------------
	// COM Umgebung aufrumen und Thread beenden
	//-----------------------------------------------------------------
	CoUninitialize();

	return bOK;
}


//===================================================================
// Diese Funktion wird aus VFP Anwendungen aufgerufen, um einen neuen
// Thread zu erzeugen. An den Thread wird eine Zeichenkette berge-
// ben. Zurckgegeben wird die Thread ID und der Handle auf den
// Thread.
//
// Alle Zeichenketten sind Unicode. Sie mssen vor dem Aufruf mit
// STRCONV() konvertiert werden.
//===================================================================
__declspec(dllexport) HANDLE CreateThreadWithObject( 
		LPWSTR lpszClass,
		LPWSTR lpszMethod,
		IDispatch *pObject, 
		LPDWORD lpdwThreadId 
	)
{
	//-----------------------------------------------------------------
	// An einen Thread knnen wir nur einen Parameter bergeben. Daher
	// speichern wir alle Werte in einer Struktur. Diese Struktur 
	// mssen wir dynamisch allokieren, damit der Speicher noch verfg-
	// bar ist, wenn diese Funktion beendet wurde. Der Speicher wird
	// durch den Thread freigegeben.
	//-----------------------------------------------------------------
	CREATETHREADWITHOBJECT *pParam;
	pParam = new CREATETHREADWITHOBJECT;
	pParam->strClass = lpszClass;
	pParam->strMethod = lpszMethod;

	//-----------------------------------------------------------------
	// Objektreferenzen knnen nicht direkt weitergeben werden. Statt-
	// dessen mssen wir einen Prozess namens Marshaling anstoen, 
	// durch den eine Objektreferenz weitergegeben werden kann. Da wir
	// auerdem sicherstellen mssen, dass das Objekt von VFP nicht 
	// freigegeben wird, bevor wir darauf zugreifen konnten, mssen wir
	// noch den Referenzzhler erhhen.
	//-----------------------------------------------------------------
	pObject->AddRef();
	HRESULT hr = CoMarshalInterThreadInterfaceInStream( 
		IID_IDispatch,
		pObject,
		&(pParam->pStream)
	);
	
	//-----------------------------------------------------------------
	// Thread erzeugen. Das Laden des COM Objektes passiert im neuen
	// Thread.
	//-----------------------------------------------------------------
	HANDLE hThread = NULL;
	if( SUCCEEDED(hr) ) {
		DWORD dwThreadID;
		hThread = CreateThread( 
			NULL,	0,CreateThreadWithObject, pParam,	0, &dwThreadID
		);
		*lpdwThreadId = dwThreadID;
	}
	
	//-----------------------------------------------------------------
	// Falls der Thread nicht erzeugt werden konnte, mssen wir den
	// Parameter wieder freigeben.
	//-----------------------------------------------------------------
	if( SUCCEEDED(hr) && hThread == NULL ) {
		delete pParam;
	}

	return hThread;
}




BOOL APIENTRY DllMain( HANDLE hModule, 
                       DWORD  ul_reason_for_call, 
                       LPVOID lpReserved
					 )
{
    return TRUE;
}

